<?php

namespace WPGraphQL\Extensions\Polylang;

use WPGraphQL\Data\Connection\MenuItemConnectionResolver;
use WPGraphQL\Data\Connection\AbstractConnectionResolver;

class MenuItem
{
    /**
     * Convert menu location to match the one generated by Polylang
     *
     * Ex. TOP_MENU -> TOP_MENU___fi
     */
    static function translate_menu_location(
        string $location,
        string $language
    ): string {
        return "${location}___${language}";
    }

    function init()
    {
        $this->create_nav_menu_locations();

        add_action(
            'graphql_register_types',
            [$this, '__action_graphql_register_types'],
            10,
            0
        );

        add_filter(
            'graphql_connection_query_args',
            [$this, '__filter_graphql_connection_query_args'],
            10,
            2
        );
    }

    function __filter_graphql_connection_query_args(
        array $query_args,
        AbstractConnectionResolver $resolver
    ) {
        if (!($resolver instanceof MenuItemConnectionResolver)) {
            return $query_args;
        }

        $args = $resolver->getArgs();

        if (!isset($args['where']['language'])) {
            return $query_args;
        }

        if (!isset($args['where']['location'])) {
            return $query_args;
        }

        // Required only when using other than the default language because the
        // menu location for the default language is the original location
        if (pll_default_language('slug') === $args['where']['language']) {
            return $query_args;
        }

        // Update the 'location' arg to use translated location
        $args['where']['location'] = self::translate_menu_location(
            $args['where']['location'],
            $args['where']['language']
        );

        // XXX. This is a hack. Modify the protected "args" so we can re-execute
        // the get_query_args method with the new "location" arg
        $ref = new \ReflectionObject($resolver);
        $args_prop = $ref->getProperty('args');
        $args_prop->setAccessible(true);
        $args_prop->setValue($resolver, $args);

        return $resolver->get_query_args();
    }

    /**
     * Nav menu locations are created on admin_init with PLL_Admin but GraphQL
     * requests do not call so we must manually call it
     */
    function create_nav_menu_locations()
    {
        // graphql_init is bit early. Delay to wp_loaded so the nav_menu object is avalable
        add_action(
            'wp_loaded',
            function () {
                global $polylang;

                if (
                    property_exists($polylang, 'nav_menu') &&
                    $polylang->nav_menu
                ) {
                    $polylang->nav_menu->create_nav_menu_locations();
                }
            },
            50
        );
    }

    function __action_graphql_register_types()
    {
        register_graphql_fields('RootQueryToMenuItemConnectionWhereArgs', [
            'language' => [
                'type' => 'LanguageCodeFilterEnum',
                'description' => '',
            ],
        ]);
    }
}
